/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK is Copyright (1995) A. S. Siddiqui and G. J. Barton

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Name
 *    asd_utils.c
 * Purpose
 *    a set of general utility routines
 * Author
 *    Asim Siddiqui
 * SccsId
 *    %W%   %U%   %E%
 */

#include <rdssp.h>
#include <asd_structs.h>
#include <ase_error.h>
#include <string.h>
#include <local.h>
#include <asd_value.h>
#include <asd_utils.h>
#include <asd_make_domains.h>
#include <asd_focus_utils.h>
#include <asm_mop.h>

extern Asd_Parameters params;

/*
 * name
 *    asd_ss_per
 * purpose
 *    to get the percentage of secondary structure in the split domain
 */
bool_t
asd_ss_per(char *ss, Asd_Domain *d_to_split, int *increment)
{
    int i;
    int ss_num1;
    int ss_num2;
    int res_num1;
    int res_num2;
    float ss_per;

    ss_num1 = 0;
    ss_num2 = 0;
    res_num1 = d_to_split->end1 - d_to_split->start1 + 1;
    res_num2 = 0;

    i = d_to_split->start1 - 1;
    while (i < d_to_split->end1) {
        if (ss[i] == 'H' || ss[i] == 'E') {
            ss_num1++;
        } /*if*/
        i++;
    } /*while*/

/* see below for explanation of type 3 */
    if (d_to_split->type == 2) {
        i = d_to_split->start2  - 1;
        while (i < d_to_split->end2) {
            if (ss[i] == 'H' || ss[i] == 'E') {
                ss_num2++;
            } /*if*/
            i++;
        } /*while*/
        res_num2 = d_to_split->end2 - d_to_split->start2 + 1;
    } /*if*/

    ss_per = ((float) (ss_num1 + ss_num2)) / ((float) (res_num1 + res_num2));

    if (ss_per < params.MIN_SS_PER) {
        *increment = (res_num1 + res_num2) / params.INCREMENT_DIVIDER;
        (*increment)++;
        return(FALSE);
    } else {
        *increment = (ss_num1 + ss_num2) / params.INCREMENT_DIVIDER;
        (*increment)++;
        return(TRUE);
    } /*if*/
} /*asd_ss_per*/

/*
 * Name
 *   asd_get_corr
 * Purpose
 *   get correlation between parts of protein chain
 *   ASSUMES s1 comes before s2
 */
bool_t
asd_get_corr(Asd_Contact_Info **c1, Asd_Contact_Info **c2, struct brookn *bn, char *ss,
             int s1_s, int s1_e, int s2_s, int s2_e, float *corr_value,
             Asd_Contact_List *c_list)
{
    int s1_int;                 /* s1 internal contacts                  */
    int s2_int;                 /* s2 internal contacts                  */
    int s1_s2_cont;             /* s1-s2 contacts                        */
    int s1_s2_sht_cont;         /* s1-s2 sheet contacts                  */
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int dummy_i;                /* dummy integer                         */
    int num_contacts;           /* number of contacts for that item      */
    float min_peak;             /* min cut off value                     */
    bool_t ss_only;             /* whether to use ss only                */
    Asd_Domain d_to_corr;       /* domain to correlate                   */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;

    s1_int = 0;
    s2_int = 0;
    s1_s2_cont = 0;
    s1_s2_sht_cont = 0;

    d_to_corr.start1 = s1_s;
    d_to_corr.end1 = s1_e;
    d_to_corr.start2 = s2_s;
    d_to_corr.end2 = s2_e;
    d_to_corr.type = 2;

    ss_only = asd_ss_per(ss, &d_to_corr, &dummy_i);

/* first scan s1 to work out s1_int and the s1-s2 contacts */
    i = s1_s;
    cf1_ptr = (contact_info + s1_s);
    cbf1_ptr = (beta_l + s1_s);
    while (i <= s1_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_s && (*cf2_ptr).res_num <= s1_e) {
                    s1_int++;
                } else if ((*cf2_ptr).res_num >= s2_s &&
                           (*cf2_ptr).res_num <= s2_e) {
                    s1_s2_cont++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num <= s2_e) {
                    s1_s2_sht_cont++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* no need to do reverse contacts in this case */
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
    } /*while*/
/* now count up internal contacts in s2 */
    i = s2_s;
    cf1_ptr = (contact_info + s2_s);
    while (i <= s2_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num <= s2_e) {
                    s2_int++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* no need to do reverse contacts in this case either */
        } /*if*/
        cf1_ptr = (cf1_ptr + 1);
        i++;
    } /*while*/

/* work out min peak value */
    if (s1_e == (s2_s - 1)) {
        if (ss_only) {
            min_peak = params.MIN_PEAK_SS_ONLY_C;
        } else {
            min_peak = params.MIN_PEAK_C;
        } /*if*/
    } else {
        if (ss_only) {
            min_peak = params.MIN_PEAK_SS_ONLY_DC;
        } else {
            min_peak = params.MIN_PEAK_DC;
        } /*if*/
    } /*if*/
/* if no intersegement contents return false set corr_value to arbitarily
 * large value
 */
    if (s1_s2_cont == 0) {
        *corr_value = min_peak * 2;
        return(FALSE);
    } /*if*/

    *corr_value = asd_calc_c_value(s1_int, s2_int, s1_s2_cont, s1_s2_sht_cont,
                                  s1_s, s1_e, 0, 0, s2_s, s2_e, 0, 0);

    if (*corr_value < min_peak) {
        return(TRUE);
    } else {
        return(FALSE);
    } /*if*/
} /*asd_get_corr*/

/*
 * Name
 *   asd_check_correlation
 * Purpose
 *   checks correlation of items if they are correlated returns input domain
 *   with items together and returns TRUE else returns items seperated and
 *   FALSE
 */
bool_t
asd_check_correlation(Asd_Contact_Info **contact_info,
                      Asd_Contact_Info **contact_rev_info,
                      struct brookn *bn, Asd_Domain *d_to_check, char *ss,
                      int num_domains, Asd_Contact_List *c_list)
{
    float corr_value1; /* correlation_value  */
    float corr_value2; /* correlation_value  */
    float corr_value3; /* correlation_value  */
    bool_t corr1;      /* are 1-2 correlated */
    bool_t corr2;      /* are 2-3 correlated */
    bool_t corr3;      /* are 1-3 correlated */
    int temp1;         /* temp storage       */
    int temp2;         /* temp storage       */

    if (num_domains == 2) {
/* first check ordering */
        if (!(d_to_check->start1 <= d_to_check->end1 &&
              d_to_check->end1 < d_to_check[1].start1 &&
              d_to_check[1].start1 <= d_to_check[1].end1)) {
            ase_error_fatal("asd_check_correlation", "error in ordering");
        } /*if*/
        if ((d_to_check->end1 - d_to_check->start1 + d_to_check[1].end1 -
             d_to_check[1].start1 + 2) < params.MIN_DOMAIN_SIZE) {
            return(FALSE);
        } /*if*/
/* now call routine to check correlation of pair */
        corr1 = asd_get_corr(contact_info, contact_rev_info, bn, ss,
                             d_to_check->start1, d_to_check->end1,
                             d_to_check[1].start1, d_to_check[1].end1,
                             &corr_value1, c_list);
        if (corr1) {
            d_to_check->start2 = d_to_check[1].start1;
            d_to_check->end2 = d_to_check[1].end1;
            return(TRUE);
        } else {
            return(FALSE);
        } /*if*/
    } else if (num_domains == 3) {
/* first check ordering */
        if (!(d_to_check->start1 <= d_to_check->end1 &&
              d_to_check->end1 < d_to_check[1].start1 &&
              d_to_check[1].start1 <= d_to_check[1].end1 &&
              d_to_check[1].end1 < d_to_check[2].start1 &&
              d_to_check[2].start1 <= d_to_check[2].end1)) {
            ase_error_fatal("asd_check_correlation", "error in ordering");
        } /*if*/

        corr1 = asd_get_corr(contact_info, contact_rev_info, bn, ss,
                             d_to_check->start1, d_to_check->end1,
                             d_to_check[1].start1, d_to_check[1].end1,
                             &corr_value1, c_list);
        corr2 = asd_get_corr(contact_info, contact_rev_info, bn, ss,
                             d_to_check[1].start1, d_to_check[1].end1,
                             d_to_check[2].start1, d_to_check[2].end1,
                             &corr_value2, c_list);
        corr3 = asd_get_corr(contact_info, contact_rev_info, bn, ss,
                             d_to_check->start1, d_to_check->end1,
                             d_to_check[2].start1, d_to_check[2].end1,
                             &corr_value3, c_list);

        if ((d_to_check->end1 - d_to_check->start1 + d_to_check[1].end1 -
             d_to_check[1].start1 + 2) < params.MIN_DOMAIN_SIZE) {
            corr1 = FALSE;
            corr1 = corr1 + corr2 + corr3;
        } /*if*/
        if ((d_to_check->end1 - d_to_check->start1 + d_to_check[2].end1 -
             d_to_check[2].start1 + 2) < params.MIN_DOMAIN_SIZE) {
            corr3 = FALSE;
            corr3 = corr1 + corr2 + corr3;
        } /*if*/
        if ((d_to_check[2].end1 - d_to_check[2].start1 + d_to_check[1].end1 -
             d_to_check[1].start1 + 2) < params.MIN_DOMAIN_SIZE) {
            corr2 = FALSE;
            corr2 = corr1 + corr2 + corr3;
        } /*if*/

        if (!corr1 && !corr2 && !corr3) {
            return(FALSE);
        } else if (corr_value1 < corr_value2 && corr_value1 < corr_value3) {
            d_to_check->start2 = d_to_check[1].start1;
            d_to_check->end2 = d_to_check[1].end1;
            d_to_check[1].start1 = d_to_check[2].start1;
            d_to_check[1].end1 = d_to_check[2].end1;
            return(TRUE);
        } else if (corr_value2 < corr_value3 && corr_value2 < corr_value1) {
            temp1 = d_to_check->start1;
            temp2 = d_to_check->end1;
            d_to_check->start1 = d_to_check[1].start1;
            d_to_check->end1 = d_to_check[1].end1;
            d_to_check->start2 = d_to_check[2].start1;
            d_to_check->end2 = d_to_check[2].end1;
            d_to_check[1].start1 = temp1;
            d_to_check[1].end1 = temp2;
            return(TRUE);
        } else if (corr_value3 < corr_value2 && corr_value3 < corr_value1) {
            d_to_check->start2 = d_to_check[2].start1;
            d_to_check->end2 = d_to_check[2].end1;
            return(TRUE);
        } /*if*/
    } else {
        ase_error_fatal("ase_check_correlation", "wrong number of domains");
    } /*if*/
} /*asd_check_correlation*/

/*
 * Name
 *   asd_get_sc_corr
 * Purpose
 *   get correlation between parts of protein chain
 *   ASSUMES s1 comes before s2
 */
bool_t
asd_get_sc_corr(Asd_Contact_Info **c1,
                Asd_Contact_Info **c2,
                struct brookn *bn, char *ss,
                int s1_s, int s1_e, int s2_s, int s2_e, float *corr_value,
                bool_t ss_only, Asd_Seg_Cont *sc_seg_info, Asd_Contact_List *c_list)
{
    int s1_int;                 /* s1 internal contacts                  */
    int s2_int;                 /* s2 internal contacts                  */
    int s1_s2_cont;             /* s1-s2 contacts                        */
    int s1_s2_sht_cont;         /* s1-s2 sheet contacts                  */
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int num_contacts;           /* number of contacts for that item      */
    float min_peak;             /* minimum peak value                    */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr;  /* temporary pointer to forward contacts */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;


    s1_int = 0;
    s2_int = 0;
    s1_s2_cont = 0;
    s1_s2_sht_cont = 0;

/* first scan s1 to work out s1_int and the s1-s2 contacts */
    i = s1_s;
    cf1_ptr = (contact_info + s1_s);
    cbf1_ptr = (beta_l + s1_s);
    while (i <= s1_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_s && (*cf2_ptr).res_num <= s1_e) {
                    s1_int++;
                } else if ((*cf2_ptr).res_num >= s2_s &&
                       (*cf2_ptr).res_num <= s2_e) {
                    s1_s2_cont++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_s &&
                    (*cf2_ptr).res_num <= s2_e) {
                    s1_s2_sht_cont++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* no need to do reverse contacts in this case */
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
    } /*while*/
/* now count up internal contacts in s2 */
    i = s2_s;
    cf1_ptr = (contact_info + s2_s);
    while (i <= s2_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num <= s2_e) {
                    s2_int++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* no need to do reverse contacts in this case either */
        } /*if*/
        cf1_ptr = (cf1_ptr + 1);
        i++;
    } /*while*/

/* work out min peak value */
    if (s1_e == (s2_s - 1)) {
        if (ss_only) {
            min_peak = params.MIN_PEAK_SS_ONLY_C;
        } else {
            min_peak = params.MIN_PEAK_C;
        } /*if*/
    } else {
        if (ss_only) {
            min_peak = params.MIN_PEAK_SS_ONLY_DC;
        } else {
            min_peak = params.MIN_PEAK_DC;
        } /*if*/
    } /*if*/
/* if no intersegement contents return false set corr_value to arbitarily
 * large value
 */
    if (s1_s2_cont == 0) {
        *corr_value = min_peak * 2;
        return(FALSE);
    } /*if*/

    *corr_value = asd_calc_c_value(s1_int, s2_int, s1_s2_cont, s1_s2_sht_cont,
                                  s1_s, s1_e, 0, 0, s2_s, s2_e, 0, 0);

    if (*corr_value < min_peak) {
        sc_seg_info->s1_int = s1_int;
        sc_seg_info->s2_int = s2_int;
        sc_seg_info->s1_s2_cont = s1_s2_cont;
        sc_seg_info->s1_s2_sht_cont = s1_s2_sht_cont;
        return(TRUE);
    } else {
        return(FALSE);
    } /*if*/
} /*asd_get_sc_corr*/

/*
 * Name
 *   asd_check_sc_correlation
 * Purpose
 *   checks correlation of items if they are correlated returns input domain
 *   with items together and returns TRUE else returns items seperated and
 *   FALSE
 *   takes account in changes to segment c
 */
bool_t
asd_check_sc_correlation(Asd_Contact_Info **contact_info,
                         Asd_Contact_Info **contact_rev_info,
                         struct brookn *bn, Asd_Domain *d_to_check,
                         int num_domains, Asd_Seg_Cont *sc_seg_info,
                         char *ss, bool_t ss_only, Asd_Contact_List *c_list)
{
    float corr_value1; /* correlation_value  */
    float corr_value2; /* correlation_value  */
    float corr_value3; /* correlation_value  */
    bool_t corr1;      /* are 1-2 correlated */
    bool_t corr2;      /* are 2-3 correlated */
    bool_t corr3;      /* are 1-3 correlated */
    int temp1;         /* temp storage       */
    int temp2;         /* temp storage       */
    int size1;
    int size2;

    if (num_domains == 2) {
/* first check ordering */
        if (!(d_to_check->start1 <= d_to_check->end1 &&
              d_to_check->end1 < d_to_check[1].start1 &&
              d_to_check[1].start1 <= d_to_check[1].end1)) {
            ase_error_fatal("asd_check_correlation", "error in ordering");
        } /*if*/
/* now call routine to check correlation of pair */
        size1 = d_to_check->end1 - d_to_check->start1 + 1;
        size2 = d_to_check[1].end1 - d_to_check[1].start1 + 1;
        if ((size1 + size2) < params.MIN_DOMAIN_SIZE) {
            return(FALSE);
        } /*if*/
        corr1 = asd_get_sc_corr(contact_info, contact_rev_info, bn, ss,
                                d_to_check->start1, d_to_check->end1,
                                d_to_check[1].start1, d_to_check[1].end1,
                                &corr_value1, ss_only, sc_seg_info, c_list);
        if (corr1) {
            d_to_check->start2 = d_to_check[1].start1;
            d_to_check->end2 = d_to_check[1].end1;
            return(TRUE);
        } else {
            return(FALSE);
        } /*if*/
    } else if (num_domains == 3) {
/* first check ordering */
        if (!(d_to_check->start1 <= d_to_check->end1 &&
              d_to_check->end1 < d_to_check[1].start1 &&
              d_to_check[1].start1 <= d_to_check[1].end1 &&
              d_to_check[1].end1 < d_to_check[2].start1 &&
              d_to_check[2].start1 <= d_to_check[2].end1)) {
            ase_error_fatal("asd_check_correlation", "error in ordering");
        } /*if*/

/* as checkin sc correlation only check pais 1,3 and also 2,3 */
        corr2 = asd_get_sc_corr(contact_info, contact_rev_info, bn, ss,
                                d_to_check[1].start1, d_to_check[1].end1,
                                d_to_check[2].start1, d_to_check[2].end1,
                                &corr_value2, ss_only, sc_seg_info, c_list);
        corr3 = asd_get_sc_corr(contact_info, contact_rev_info, bn, ss,
                                d_to_check->start1, d_to_check->end1,
                                d_to_check[2].start1, d_to_check[2].end1,
                                &corr_value3, ss_only, sc_seg_info, c_list);

        if (!corr2 && !corr3) {
            return(FALSE);
        } else if (corr_value2 < corr_value3) {
            temp1 = d_to_check->start1;
            temp2 = d_to_check->end1;
            d_to_check->start1 = d_to_check[1].start1;
            d_to_check->end1 = d_to_check[1].end1;
            d_to_check->start2 = d_to_check[2].start1;
            d_to_check->end2 = d_to_check[2].end1;
            d_to_check[1].start1 = temp1;
            d_to_check[1].end1 = temp2;
            size1 = d_to_check->end1 - d_to_check->start1 + 1;
            size2 = d_to_check->end2 - d_to_check->start2 + 1;
            if ((size1 + size2) < params.MIN_DOMAIN_SIZE) {
                return(FALSE);
            } /*if*/
            (void) asd_get_sc_corr(contact_info, contact_rev_info, bn, ss,
                                   d_to_check->start1, d_to_check->end1,
                                   d_to_check->start2, d_to_check->end2,
                                   &corr_value1, ss_only, sc_seg_info, c_list);
            return(TRUE);
        } else {
            d_to_check->start2 = d_to_check[2].start1;
            d_to_check->end2 = d_to_check[2].end1;
            size1 = d_to_check->end1 - d_to_check->start1 + 1;
            size2 = d_to_check->end2 - d_to_check->start2 + 1;
            if ((size1 + size2) < params.MIN_DOMAIN_SIZE) {
                return(FALSE);
            } /*if*/
            return(TRUE);
        } /*if*/
    } else {
        ase_error_fatal("ase_check_correlation", "wrong number of domains");
    } /*if*/
} /*asd_check_sc_correlation*/

/* Name
 *    asd_total_contacts
 * Purpose
 *   to work out the total number of contacts in  given segments
 *   and create the lists of contacts
 */
Asd_Contact_List *
asd_total_contacts(Asd_Domain *d_to_calc, char *ss, bool_t ss_only,
                   Asd_Contact_Info **contact_info,
                   Asd_Contact_Info **contact_rev_info, struct brookn *bn,
                   int *total_contacts)
{
    int num_atoms;              /* number of atoms that are being looked at */ 
    int i;                      /* loop counter                             */
    int j;                      /* loop counter                             */
    int num_contacts;           /* number of contacts for that item         */
    int nc;                     /* number of contacts for item              */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts    */
    Asd_Contact_Info *cf2_ptr;  /* temporary pointer to forward contacts    */
    Asd_Contact_List *c_list;   /* list of contacts                         */

    *total_contacts = 0;
    if (d_to_calc->type == 2) {
        num_atoms = d_to_calc->end2;
    } else {
        num_atoms = d_to_calc->end1;
    } /*if*/
    c_list = (Asd_Contact_List *) asm_malloc(sizeof(Asd_Contact_List));
    c_list->cl = (Asd_Contact_Info **) asm_malloc(sizeof(Asd_Contact_Info *) * 
                                                        (num_atoms + 1));
    c_list->rcl = (Asd_Contact_Info **) asm_malloc(sizeof(Asd_Contact_Info *) * 
                                                        (num_atoms + 1));
    c_list->bl = (Asd_Contact_Info **) asm_malloc(sizeof(Asd_Contact_Info *) * 
                                                        (num_atoms + 1));
    c_list->rbl = (Asd_Contact_Info **) asm_malloc(sizeof(Asd_Contact_Info *) * 
                                                        (num_atoms + 1));

    i = 0;
    while (i <= num_atoms) {
        c_list->cl[i] = (Asd_Contact_Info *)
                         asm_malloc(sizeof(Asd_Contact_Info));
        c_list->cl[i][0].res_num = 0;
        c_list->rcl[i] = (Asd_Contact_Info *)
                          asm_malloc(sizeof(Asd_Contact_Info));
        c_list->rcl[i][0].res_num = 0;
        c_list->bl[i] = (Asd_Contact_Info *)
                         asm_malloc(sizeof(Asd_Contact_Info));
        c_list->bl[i][0].res_num = 0;
        c_list->rbl[i] = (Asd_Contact_Info *)
                          asm_malloc(sizeof(Asd_Contact_Info));
        c_list->rbl[i][0].res_num = 0;
        i++;
    } /*while*/

    i = d_to_calc->start1;
    cf1_ptr = (contact_info + i);
    while (i <= d_to_calc->end1) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((!ss_only || (ss_only && (ss[(*cf2_ptr).res_num - 1] == 'H'
                              || ss[(*cf2_ptr).res_num - 1] == 'E'))) &&
                   (((*cf2_ptr).res_num >= d_to_calc->start1 &&
                     (*cf2_ptr).res_num <= d_to_calc->end1) ||
                    ((*cf2_ptr).res_num >= d_to_calc->start2 &&
                     (*cf2_ptr).res_num <= d_to_calc->end2 &&
                      d_to_calc->type == 2))) {
                    if ((*cf2_ptr).ctype != 'E') {
                        (*total_contacts)++;
/* add forward contacts */
                        nc = c_list->cl[i][0].res_num;
                        nc++;
                        c_list->cl[i] = (Asd_Contact_Info *)
                                         asm_realloc(c_list->cl[i],
                                         sizeof(Asd_Contact_Info) * (nc + 1));
                        c_list->cl[i][0].res_num = nc;
                        c_list->cl[i][nc].res_num = (*cf2_ptr).res_num;
/* add reverse contacts */
                        nc = c_list->rcl[(*cf2_ptr).res_num][0].res_num;
                        nc++;
                        c_list->rcl[(*cf2_ptr).res_num] = (Asd_Contact_Info *)
                                    asm_realloc(c_list->rcl[(*cf2_ptr).res_num],
                                    sizeof(Asd_Contact_Info) * (nc + 1));
                        c_list->rcl[(*cf2_ptr).res_num][0].res_num = nc;
                        c_list->rcl[(*cf2_ptr).res_num][nc].res_num = i;
                    } else {
/* add forward contacts */
                        nc = c_list->bl[i][0].res_num;
                        nc++;
                        c_list->bl[i] = (Asd_Contact_Info *)
                                         asm_realloc(c_list->bl[i],
                                         sizeof(Asd_Contact_Info) * (nc + 1));
                        c_list->bl[i][0].res_num = nc;
                        c_list->bl[i][nc].res_num = (*cf2_ptr).res_num;
/* add reverse contacts */
                        nc = c_list->rbl[(*cf2_ptr).res_num][0].res_num;
                        nc++;
                        c_list->rbl[(*cf2_ptr).res_num] = (Asd_Contact_Info *)
                                   asm_realloc(c_list->rbl[(*cf2_ptr).res_num],
                                   sizeof(Asd_Contact_Info) * (nc + 1));
                        c_list->rbl[(*cf2_ptr).res_num][0].res_num = nc;
                        c_list->rbl[(*cf2_ptr).res_num][nc].res_num = i;
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
    } /*while*/

    if (d_to_calc->type == 2) {
        i = d_to_calc->start2;
        cf1_ptr = (contact_info + i);
        while (i <= d_to_calc->end2) {
            if (!ss_only ||
                (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
                cf2_ptr = *cf1_ptr;
                num_contacts = (*cf2_ptr).res_num;
                cf2_ptr++;
                j = 0;
                while (j < num_contacts) {
                    if ((!ss_only || (ss_only &&
                                    (ss[(*cf2_ptr).res_num - 1] == 'H' ||
                                     ss[(*cf2_ptr).res_num - 1] == 'E'))) &&
                       (((*cf2_ptr).res_num >= d_to_calc->start1 &&
                         (*cf2_ptr).res_num <= d_to_calc->end1) ||
                        ((*cf2_ptr).res_num >= d_to_calc->start2 &&
                         (*cf2_ptr).res_num <= d_to_calc->end2))) {
                        if ((*cf2_ptr).ctype != 'E') {
                            (*total_contacts)++;
/* add forward contacts */
                            nc = c_list->cl[i][0].res_num;
                            nc++;
                            c_list->cl[i] = (Asd_Contact_Info *)
                                           asm_realloc(c_list->cl[i],
                                           sizeof(Asd_Contact_Info) * (nc + 1));
                            c_list->cl[i][0].res_num = nc;
                            c_list->cl[i][nc].res_num = (*cf2_ptr).res_num;
/* add reverse contacts */
                            nc = c_list->rcl[(*cf2_ptr).res_num][0].res_num;
                            nc++;
                            c_list->rcl[(*cf2_ptr).res_num] =
                                   (Asd_Contact_Info *)
                                    asm_realloc(c_list->rcl[(*cf2_ptr).res_num],
                                    sizeof(Asd_Contact_Info) * (nc + 1));
                            c_list->rcl[(*cf2_ptr).res_num][0].res_num = nc;
                            c_list->rcl[(*cf2_ptr).res_num][nc].res_num = i;
                        } else {
/* add forward contacts */
                            nc = c_list->bl[i][0].res_num;
                            nc++;
                            c_list->bl[i] = (Asd_Contact_Info *)
                                           asm_realloc(c_list->bl[i],
                                           sizeof(Asd_Contact_Info) * (nc + 1));
                            c_list->bl[i][0].res_num = nc;
                            c_list->bl[i][nc].res_num = (*cf2_ptr).res_num;
/* add reverse contacts */
                            nc = c_list->rbl[(*cf2_ptr).res_num][0].res_num;
                            nc++;
                            c_list->rbl[(*cf2_ptr).res_num] =
                                   (Asd_Contact_Info *)
                                    asm_realloc(c_list->rbl[(*cf2_ptr).res_num],
                                    sizeof(Asd_Contact_Info) * (nc + 1));
                            c_list->rbl[(*cf2_ptr).res_num][0].res_num = nc;
                            c_list->rbl[(*cf2_ptr).res_num][nc].res_num = i;
                        } /*if*/
                    } /*if*/
                    j++;
                    cf2_ptr++;
                } /*while*/
            } /*if*/
            i++;
            cf1_ptr = (cf1_ptr + 1);
        } /*while*/
    } /*if*/

    return(c_list);
} /*asd_total_contacts*/

/*
 * Name
 *   asd_focus_range
 * Purpose
 *   for large proteins that had incremented cuts find exact position of largest
 *   value
 *   AT THE MOMENT only averages positions with plateau maximums
 */
void
asd_focus_range(Asd_Domain *d_to_focus, Asd_Domain *d_to_split,
                Asd_Contact_Info **contact_info,
                Asd_Contact_Info **contact_rev_info, struct brookn *bn,
                int start_chain, int end_chain, char *ss)
{
    d_to_focus->start1 = 0;
    d_to_focus->end1 = 0;
    d_to_focus->start2 = 0;
    d_to_focus->end2 = 0;

    if (d_to_focus->range.s1_s_max == d_to_focus->range.s1_s_min) {
        d_to_focus->start1 = d_to_focus->range.s1_s_max;
    } /*if*/ 
    if (d_to_focus->range.s1_e_max == d_to_focus->range.s1_e_min) {
        d_to_focus->end1 = d_to_focus->range.s1_e_max;
    } /*if*/ 
    if (d_to_focus->type == 2) {
        if (d_to_focus->range.s2_s_max == d_to_focus->range.s2_s_min) {
            d_to_focus->start2 = d_to_focus->range.s2_s_max;
        } /*if*/ 
        if (d_to_focus->range.s2_e_max == d_to_focus->range.s2_e_min) {
            d_to_focus->end2 = d_to_focus->range.s2_e_max;
        } /*if*/ 
    } /*if*/

    if (d_to_focus->start1 != 0) {
        d_to_focus->range.s1_s_min = d_to_focus->start1;
        d_to_focus->range.s1_s_max = d_to_focus->start1;
    } /*if*/
    if (d_to_focus->end1 != 0) {
        d_to_focus->range.s1_e_min = d_to_focus->end1;
        d_to_focus->range.s1_e_max = d_to_focus->end1;
    } /*if*/
    if (d_to_focus->start2 != 0) {
        d_to_focus->range.s2_s_min = d_to_focus->start2;
        d_to_focus->range.s2_s_max = d_to_focus->start2;
    } /*if*/
    if (d_to_focus->end2 != 0) {
        d_to_focus->range.s2_e_min = d_to_focus->end2;
        d_to_focus->range.s2_e_max = d_to_focus->end2;
    } /*if*/

    if (d_to_focus->start1 == 0 || d_to_focus->end1 == 0 ||
      ((d_to_focus->start2 == 0 || d_to_focus->end2 == 0) &&
        d_to_focus->type == 2)) {
        asd_narrow_range(d_to_focus, d_to_split, contact_info, contact_rev_info,
                         bn, start_chain, end_chain, ss);
    } /*if*/
} /*asd_focus_range*/

/*
 * name
 *    asd_free_c_list
 * purpose
 *    free items in c_list
 */
void
asd_free_c_list(Asd_Domain *d_to_free, Asd_Contact_List *c_list)
{
    int i;
    int end;

    if (d_to_free->type == 2) {
        end = d_to_free->end2;
    } else {
        end = d_to_free->end1;
    } /*if*/

    i = 0;
    while (i <= end) {
        free(c_list->cl[i]);
        free(c_list->rcl[i]);
        free(c_list->bl[i]);
        free(c_list->rbl[i]);
        i++;
    } /*while*/

    free(c_list->cl);
    free(c_list->rcl);
    free(c_list->bl);
    free(c_list->rbl);
    
    free(c_list);
} /*asd_free_c_list*/

/*
 * name
 *    asd_complete_corr
 * purpose
 *    to carry out a complete correlation check on two domains
 */
float
asd_complete_corr(Asd_Contact_Info **contact_info,
                  Asd_Contact_Info **contact_rev_info,
                  struct brookn *bn, char *ss, Asd_Domain_List *d_list,
                  int d1, int d2)
{
    int s1_int;                 /* s1 internal contacts                  */
    int s2_int;                 /* s2 internal contacts                  */
    int s1_s2_cont;             /* s1-s2 contacts                        */
    int s1_s2_sht_cont;         /* s1-s2 sheet contacts                  */
    int i;                      /* loop counter                          */
    int dum1, dum2;
    float int_1_f;
    float int_2_f;
    float ext_no_f;

    s1_int = 0;
    s2_int = 0;
    s1_s2_cont = 0;
    s1_s2_sht_cont = 0;

/* first scan d1 to find internal and external contacts */
    asd_scan_domain(contact_info, contact_rev_info, bn, ss, d_list, d1, d2,
                    &(d_list->domains[d1]), &s1_int, &s1_s2_cont,
                    &s1_s2_sht_cont, TRUE);

    i = 0;
    while (i < d_list->n_d_list) {
        if (d_list->domains[i].d_assigned == d1 && i != d1) {
            asd_scan_domain(contact_info, contact_rev_info, bn, ss, d_list, d1,
                            d2, &(d_list->domains[i]), &s1_int, &s1_s2_cont,
                            &s1_s2_sht_cont, TRUE);
        } /*if*/
        i++;
    } /*while*/

    i = 0;
    while (i < d_list->n_b_left_over) {
        if (d_list->bits_left_over[i].d_assigned == d1) {
            asd_scan_domain(contact_info, contact_rev_info, bn, ss, d_list, d1,
                            d2, &(d_list->bits_left_over[i]), &s1_int,
                            &s1_s2_cont, &s1_s2_sht_cont, TRUE);
        } /*if*/
        i++;
    } /*while*/

/* now get internals of d2 */
    asd_scan_domain(contact_info, contact_rev_info, bn, ss, d_list, d2, d1,
                    &(d_list->domains[d2]), &s2_int, &dum1,
                    &dum2, FALSE);

    i = 0;
    while (i < d_list->n_d_list) {
        if (d_list->domains[i].d_assigned == d2 && i != d2) {
            asd_scan_domain(contact_info, contact_rev_info, bn, ss, d_list, d2,
                            d1, &(d_list->domains[i]), &s2_int, &dum1,
                            &dum2, FALSE);
        } /*if*/
        i++;
    } /*while*/

    i = 0;
    while (i < d_list->n_b_left_over) {
        if (d_list->bits_left_over[i].d_assigned == d2) {
            asd_scan_domain(contact_info, contact_rev_info, bn, ss, d_list, d2,
                            d1, &(d_list->bits_left_over[i]), &s2_int,
                            &dum1, &dum2, FALSE);
        } /*if*/
        i++;
    } /*while*/

    if (s1_int == 0) {
        return(-1.0);
    } else {
        int_1_f = (float) s1_int;
    } /*if*/

    if (s2_int == 0) {
        return(-1.0);
    } else {
        int_2_f = (float) s2_int;
    } /*if*/

    if (s1_s2_cont == 0) {
        return(-1.0);
    } else {
        ext_no_f = (float) s1_s2_cont;
    } /*if*/

    ext_no_f = (ext_no_f * ((float) s1_s2_sht_cont) * params.E_WEIGHT) + ext_no_f;

    return(((int_1_f / ext_no_f) * (int_2_f / ext_no_f)));

} /*asd_complete_corr*/
