/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Title
 *    asd_glob_interface.c
 * Purpose
 *    contains routines that work out globularity
 * Author
 *    Asim Siddiqui
 * SccsId
 *    %W%   %U%    %E%
 */

#include <rdssp.h>
#include <asd_structs.h>
#include <asm_mop.h>
#include <ase_error.h>
#include <glob_routs.h>
#include <asd_glob_interface.h>
#include <ass_stamp_utils.h>
#include <local.h>
#include <asd_make_domains.h>
#include <ass_stamp_utils.h>
#include <stdio.h>
#include <asd_utils.h>
#include <asd_value.h>

extern Asd_Parameters params;

#define GLOB_CHECK_NONE              0
#define GLOB_CHECK_ALL               1
#define GLOB_CHECK_AVERAGE           2
#define GLOB_CHECK_NON_GLOB_BIT_LEFT 3
#define GLOB_CHECK_AT_END            4
#define GLOB_CHECK_TYPE              4

/*
 * name
 *    asd_check_domains
 * purpose
 *    to check the globularity of the domains against the system wanted
 */
void
asd_check_domains(Asd_Domain_List *d_list, struct brookn *bn, bool_t comments)
{
    int i;            /* loop counter              */
    int glob_ok;      /* whether globularity is ok */
    float total_glob; /* globularity               */

    glob_ok = 1;
    total_glob = 0.0;
    if (d_list == NULL || GLOB_CHECK_TYPE == GLOB_CHECK_NONE ||
                          GLOB_CHECK_TYPE == GLOB_CHECK_AT_END) {
        return;
    } else if (GLOB_CHECK_TYPE == GLOB_CHECK_ALL ||
               GLOB_CHECK_TYPE == GLOB_CHECK_NON_GLOB_BIT_LEFT) {
        if ((d_list->n_d_list > 1 || (d_list->n_b_left_over > 0 &&
             d_list->n_d_list > 0)) && comments) {
            ass_glob_comments(1, d_list->d_split_up, bn);
        } /*if*/
        i = 0;
        while (i < d_list->n_d_list) {
            d_list->domains[i].d_assigned = 0;
            if (d_list->domains[i].glob < (LARGE_NEG_GLOB + 1.0)) {
                d_list->domains[i].glob = as_globularity(
                            ass_write_domain_string(&(d_list->domains[i]), bn));
            } /*if*/
            if (d_list->domains[i].glob > params.MAX_ALLOWABLE_GLOB) {
                d_list->domains[i].d_assigned = -1;
                glob_ok = 0;
            } /*if*/
            if ((d_list->n_d_list > 1 || (d_list->n_b_left_over > 0 &&
                 d_list->n_d_list > 0)) && comments) {
                ass_glob_comments(2, &(d_list->domains[i]), bn);
            } /*if*/
            i++;
        } /*while*/
    } else if (GLOB_CHECK_TYPE == GLOB_CHECK_AVERAGE) {
        i = 0;
        while (i < d_list->n_d_list) {
            if (d_list->domains[i].glob < (LARGE_NEG_GLOB + 1.0)) {
                d_list->domains[i].glob = as_globularity(
                            ass_write_domain_string(&(d_list->domains[i]), bn));
            } /*if*/
            if ((d_list->n_d_list > 1 || (d_list->n_b_left_over > 0 &&
                 d_list->n_d_list > 0)) && comments) {
                ass_glob_comments(2, &(d_list->domains[i]), bn);
            } /*if*/
            total_glob = total_glob + d_list->domains[i].glob;
            i++;
        } /*while*/
        total_glob = total_glob / (float) i;
        if (total_glob > params.MAX_ALLOWABLE_GLOB) {
            glob_ok = 0;
        } /*if*/
    } else {
        ase_error_fatal("asd_check_domains", "unrecognisable GLOB_CHECK_TYPE");
    } /*if*/

    if (glob_ok == 0 && GLOB_CHECK_TYPE != GLOB_CHECK_NON_GLOB_BIT_LEFT) {
        if (GLOB_CHECK_TYPE == GLOB_CHECK_ALL) {
            if ((d_list->n_d_list > 1 || (d_list->n_b_left_over > 0 &&
                 d_list->n_d_list > 0)) && comments) {
                ass_glob_comments(4, d_list->d_split_up, bn);
            } /*if*/
        } else if (GLOB_CHECK_TYPE == GLOB_CHECK_AVERAGE) {
            if ((d_list->n_d_list > 1 || (d_list->n_b_left_over > 0 &&
                 d_list->n_d_list > 0)) && comments) {
                ass_glob_comments(3, d_list->d_split_up, bn);
            } /*if*/
        } /*if*/
        d_list->n_d_list = 1;
        d_list->domains = d_list->d_split_up;
        d_list->domains->glob = LARGE_NEG_GLOB;
        d_list->n_b_left_over = 0;
    } /*if*/
} /*asd_check_domains*/

/*
 * name
 *    asd_check_glob_end
 * purpose
 *    to do checks of globularity at the end os the algorithm
 */
void
asd_check_glob_end(Asd_Domain *d_to_split, Asd_Domain_List *d_list,
                   Asd_Contact_Info **contact_info,
                   Asd_Contact_Info **contact_rev_info,
                   struct brookn *bn, char *ss)
{
    int i, j, k;
    float glob;
    float corr_value;
    float new_corr_value;
    int found_it;
    bool_t again;

    if (GLOB_CHECK_TYPE != GLOB_CHECK_AT_END) {
        return;
    } /*if*/

    again = FALSE;
    ass_glob_comments(1, d_to_split, bn);
    i = 0;
    while (i < d_list->n_d_list) {
        if (d_list->domains[i].d_assigned == -2) {
            glob = as_globularity(
                                ass_write_domain_defn_with_all(d_list, i, bn));
            d_list->domains[i].glob = glob;
            ass_glob_comments(2, &(d_list->domains[i]), bn);

            if (glob > params.MAX_ALLOWABLE_GLOB) {
                 d_list->domains[i].d_assigned = -1;
                 ass_glob_comments(5, &(d_list->domains[i]), bn);
                 corr_value = -1.0;
                 j = 0;
                 found_it = 0;
                 while (j < d_list->n_d_list) {
                     if (i != j && d_list->domains[j].d_assigned == -2) {
                         found_it = 1;
                         new_corr_value = asd_complete_corr(contact_info,
                                      contact_rev_info, bn, ss, d_list, i, j);
                         if (corr_value < 0.0 || (new_corr_value < corr_value &&
                             new_corr_value >= 0.0)) {
                             corr_value = new_corr_value;
                             d_list->domains[i].d_assigned = j;
                         } /*if*/
                     } /*if*/
                     j++;
                 } /*while*/
                 if (!found_it) {
                     d_list->domains[i].d_assigned = -2;
                 } /*if*/
                 if (corr_value >= 0.0) {
                     j = d_list->domains[i].d_assigned;
                     ass_glob_comments(6, &(d_list->domains[j]), bn);
                     k = 0;
                     while (k < d_list->n_b_left_over) {
                         if (d_list->bits_left_over[k].d_assigned == i) {
                             d_list->bits_left_over[k].d_assigned = j;
                         } /*if*/
                         k++;
                     } /*while*/
                     k = 0;
                     while (k < d_list->n_d_list) {
                         if (d_list->domains[k].d_assigned == i) {
                             d_list->domains[k].d_assigned = j;
                         } /*if*/
                         k++;
                     } /*while*/
                     again = TRUE;
                 } /*if*/
             } /*if*/
         } /*if*/
         i++;
    } /*while*/

    if (again) {
        asd_check_glob_end(d_to_split, d_list, contact_info,
                           contact_rev_info, bn, ss);
    } /*if*/
} /*asd_check_glob_end*/


/*
 * name
 *    asd_copy_d_list
 * purpose
 *    to copy items in one d_list to the end of the other
 *    ASSUMES space has already been allocated in d_list_f
 */
void
asd_copy_d_list(Asd_Domain_List *d_list_f, int n_domains, int n_b_left_over,
                Asd_Domain_List *d_list)
{
    int i; /* loop counter */
    int j; /* loop counter */

    i = n_domains - d_list->n_d_list;
    j = 0;
    while (i < n_domains) {
        d_list_f->domains[i] = d_list->domains[j];
        i++;
        j++;
    } /*while*/
    if (j > d_list->n_d_list) {
        ase_error_fatal("asd_copy_d_list", "not enough room");
    } /*if*/

    i = n_b_left_over - d_list->n_b_left_over;
    j = 0;
    while (i < n_b_left_over) {
        d_list_f->bits_left_over[i] = d_list->bits_left_over[j];
        i++;
        j++;
    } /*while*/
    if (j > d_list->n_b_left_over) {
        ase_error_fatal("asd_copy_d_list", "not enough room");
    } /*if*/
} /*asd_copy_d_list*/

/*
 * name
 *    asd_check_globularity
 * purpose
 *    top level routine for checking globularity also returns bits as one
 */
Asd_Domain_List *
asd_check_globularity(Asd_Domain_List *d_list1, Asd_Domain_List *d_list2,
                      Asd_Domain_List *d_list3, Asd_Domain_List *d_list4,
                      struct brookn *bn, Asd_Domain *d_orig)
{
    int n_domains;             /* number of domains             */
    int n_b_left_over;         /* number of left over bits      */
    Asd_Domain_List *d_list_f; /* final d_list for this routine */

    asd_check_domains(d_list1, bn, FALSE);
    asd_check_domains(d_list2, bn, FALSE);
    asd_check_domains(d_list3, bn, FALSE);
    asd_check_domains(d_list4, bn, FALSE);

    d_list_f = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
    d_list_f->d_split_up = d_orig;
    d_list_f->d_split_up->glob = LARGE_NEG_GLOB;
    d_list_f->domains = NULL;
    d_list_f->bits_left_over = NULL;
    n_domains = 0;
    n_b_left_over = 0;

    if (d_list1 != NULL) {
        n_domains = n_domains + d_list1->n_d_list;
        d_list_f->domains = (Asd_Domain *) asm_realloc(d_list_f->domains,
                                           sizeof(Asd_Domain) * n_domains);
        n_b_left_over = n_b_left_over + d_list1->n_b_left_over;
        d_list_f->bits_left_over = (Asd_Domain *)
                                           asm_realloc(d_list_f->bits_left_over,
                                            sizeof(Asd_Domain) * n_b_left_over);
        asd_copy_d_list(d_list_f, n_domains, n_b_left_over, d_list1);
    } /*if*/
    if (d_list2 != NULL) {
        n_domains = n_domains + d_list2->n_d_list;
        d_list_f->domains = (Asd_Domain *) asm_realloc(d_list_f->domains,
                                           sizeof(Asd_Domain) * n_domains);
        n_b_left_over = n_b_left_over + d_list2->n_b_left_over;
        d_list_f->bits_left_over = (Asd_Domain *)
                                           asm_realloc(d_list_f->bits_left_over,
                                            sizeof(Asd_Domain) * n_b_left_over);
        asd_copy_d_list(d_list_f, n_domains, n_b_left_over, d_list2);
    } /*if*/
    if (d_list3 != NULL) {
        n_domains = n_domains + d_list3->n_d_list;
        d_list_f->domains = (Asd_Domain *) asm_realloc(d_list_f->domains,
                                           sizeof(Asd_Domain) * n_domains);
        n_b_left_over = n_b_left_over + d_list3->n_b_left_over;
        d_list_f->bits_left_over = (Asd_Domain *)
                                           asm_realloc(d_list_f->bits_left_over,
                                            sizeof(Asd_Domain) * n_b_left_over);
        asd_copy_d_list(d_list_f, n_domains, n_b_left_over, d_list3);
    } /*if*/
    if (d_list4 != NULL) {
        n_domains = n_domains + d_list4->n_d_list;
        d_list_f->domains = (Asd_Domain *) asm_realloc(d_list_f->domains,
                                           sizeof(Asd_Domain) * n_domains);
        n_b_left_over = n_b_left_over + d_list4->n_b_left_over;
        d_list_f->bits_left_over = (Asd_Domain *)
                                           asm_realloc(d_list_f->bits_left_over,
                                            sizeof(Asd_Domain) * n_b_left_over);
        asd_copy_d_list(d_list_f, n_domains, n_b_left_over, d_list4);
    } /*if*/

    d_list_f->n_b_left_over = n_b_left_over;
    d_list_f->n_d_list = n_domains;

    asd_check_domains(d_list_f, bn, TRUE);

    return(d_list_f);
} /*asd_check_globularity*/

float c_dens; /* contact density */

/*
 * name
 *    asd_match_up_bits
 * purpose
 *    to match up bit and see if it matches up with domain
 */
bool_t
asd_match_up_bits(Asd_Domain *d_to_compare, Asd_Domain *bit,
                  Asd_Contact_Info **contact_info,
                  Asd_Contact_Info **contact_rev_info, char *ss,
                  float *corr_value, struct brookn *bn)
{
    int s1_int;                 /* s1 internal contacts                  */
    int s2_int;                 /* s2 internal contacts                  */
    int s1_s2_cont;             /* s1-s2 contacts                        */
    int s1_s2_sht_cont;         /* s1-s2 sheet contacts                  */
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int num_contacts;           /* number of contacts for that item      */
    int dummy_i;                /* dummy integer                         */
    bool_t ss_only;             /* whether to use ss only                */
    bool_t cont;                /* continous                             */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr;  /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info *cr2_ptr;  /* temporary pointer to reverse contacts */

    s1_int = 0;
    s2_int = 0;
    s1_s2_cont = 0;
    s1_s2_sht_cont = 0;

/*    ss_only = asd_ss_per(ss, bit, &dummy_i); */
    ss_only = FALSE; 
/* first scan s1 to work out s1_int and the s1-s2 contacts */
    i = bit->start1;
    cf1_ptr = (contact_info + i);
    cr1_ptr = (contact_rev_info + i);
    while (i <= bit->end1) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if (!ss_only || (ss_only && (ss[(*cf2_ptr).res_num - 1] == 'H'
                             || ss[(*cf2_ptr).res_num - 1] == 'E'))) {
                    if ((*cf2_ptr).res_num >= bit->start1 &&
                        (*cf2_ptr).res_num <= bit->end1) {
                        if ((*cf2_ptr).ctype != 'E') {
                            s1_int++;
                        } /*if*/
                    } else if (((*cf2_ptr).res_num >= d_to_compare->start1 &&
                                (*cf2_ptr).res_num <= d_to_compare->end1) ||
                               ((*cf2_ptr).res_num >= d_to_compare->start2 &&
                                (*cf2_ptr).res_num <= d_to_compare->end2 &&
                                d_to_compare->type == 2)) {
                        if ((*cf2_ptr).ctype != 'E') {
                            s1_s2_cont++;
                        } else {
                            s1_s2_sht_cont++;
                        } /*if*/
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - no need to count internal ones again */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if (!ss_only || (ss_only && (ss[(*cr2_ptr).res_num - 1] == 'H'
                             || ss[(*cr2_ptr).res_num - 1] == 'E'))) {
                    if (((*cr2_ptr).res_num >= d_to_compare->start1 &&
                         (*cr2_ptr).res_num <= d_to_compare->end1) ||
                        ((*cr2_ptr).res_num >= d_to_compare->start2 &&
                         (*cr2_ptr).res_num <= d_to_compare->end2 &&
                         d_to_compare->type == 2)) {
                        if ((*cr2_ptr).ctype != 'E') {
                            s1_s2_cont++;
                        } else {
                            s1_s2_sht_cont++;
                        } /*if*/
                    } /*if*/
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
    } /*while*/
/* now scan s2 to work out its internal contacts assume second bit of s2
   (if present) follows first bit
 */
    i = d_to_compare->start1;
    cf1_ptr = (contact_info + i);
    while (i <= d_to_compare->end1) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if (!ss_only || (ss_only && (ss[(*cf2_ptr).res_num - 1] == 'H'
                             || ss[(*cf2_ptr).res_num - 1] == 'E'))) {
                    if (((*cf2_ptr).res_num >= d_to_compare->start1 &&
                         (*cf2_ptr).res_num <= d_to_compare->end1) ||
                        ((*cf2_ptr).res_num >= d_to_compare->start2 &&
                         (*cf2_ptr).res_num <= d_to_compare->end2 &&
                         d_to_compare->type == 2)) {
                        if ((*cf2_ptr).ctype != 'E') {
                            s2_int++;
                        } /*if*/
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
    } /*while*/

    if (d_to_compare->type == 2) {
        i = d_to_compare->start2;
        cf1_ptr = (contact_info + i);
        while (i <= d_to_compare->end2) {
            if (!ss_only ||
                (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
                cf2_ptr = *cf1_ptr;
                num_contacts = (*cf2_ptr).res_num;
                cf2_ptr++;
                j = 0;
                while (j < num_contacts) {
                    if (!ss_only ||
                        (ss_only && (ss[(*cf2_ptr).res_num - 1] == 'H'
                                  || ss[(*cf2_ptr).res_num - 1] == 'E'))) {
                        if ((*cf2_ptr).res_num >= d_to_compare->start2 &&
                            (*cf2_ptr).res_num <= d_to_compare->end2) {
                            if ((*cf2_ptr).ctype != 'E') {
                                s2_int++;
                            } /*if*/
                        } /*if*/
                    } /*if*/
                    j++;
                    cf2_ptr++;
                } /*while*/
            } /*if*/
            i++;
            cf1_ptr = (cf1_ptr + 1);
        } /*while*/
    } /*if*/

/* if no intersegement contents return false set corr_value to arbitarily
 * large value
 */
    if (s1_s2_cont == 0) {
        if (ss_only) {
            *corr_value = -1.0;
        } else {
            *corr_value = -1.0;
        } /*if*/
        return(FALSE);
    } /*if*/

    if (d_to_compare->type == 1) {
        d_to_compare->start2 = 0;
        d_to_compare->end2 = 0;
    } /*if*/

    
    if (bit->start1 == (d_to_compare->end1 + 1) ||
        bit->end1 == (d_to_compare->start1 - 1) ||
       (bit->start1 == (d_to_compare->end2 + 1) ||
        bit->end1 == (d_to_compare->start2 - 1) && d_to_compare->type == 2)) {
        cont = TRUE;
        if (bit->start1 == (d_to_compare->end1 + 1)) {
            asd_hel_reduce_ext_by(d_to_compare->end1, &s1_s2_cont);
        } /*if*/
        if (bit->end1 == (d_to_compare->start1 - 1)) {
            asd_hel_reduce_ext_by(bit->end1, &s1_s2_cont);
        } /*if*/
        if (bit->start1 == (d_to_compare->end2 + 1) &&
                            d_to_compare->type == 2) {
            asd_hel_reduce_ext_by(d_to_compare->end2, &s1_s2_cont);
        } /*if*/
        if (bit->end1 == (d_to_compare->start2 - 1) &&
                          d_to_compare->type == 2) {
            asd_hel_reduce_ext_by(bit->end1, &s1_s2_cont);
        } /*if*/
    } else {
        cont = FALSE;
    } /*if*/
    s1_int = c_dens * ((float) (bit->end1 - bit->start1 + 1));

/*    *corr_value = asd_calc_c_value(s1_int, s2_int, s1_s2_cont, s1_s2_sht_cont,
                             bit->start1, bit->end1, 0, 0,
                             d_to_compare->start1, d_to_compare->end1,
                             d_to_compare->start2, d_to_compare->end2); */

    *corr_value = asd_calc_c_value(s1_int, s1_int, s1_s2_cont, s1_s2_sht_cont,
                             bit->start1, bit->end1, 0, 0,
                             bit->start1, bit->end1, 0, 0);


    if ((((ss_only && *corr_value < params.MIN_PEAK_SS_ONLY_BLO_C) ||
         (!ss_only && *corr_value < params.MIN_PEAK_BLO_C)) && cont) ||
        (((ss_only && *corr_value < params.MIN_PEAK_SS_ONLY_BLO_DC) ||
         (!ss_only && *corr_value < params.MIN_PEAK_BLO_DC)) && !cont)) {
        return(TRUE);
    } else {
        return(FALSE);
    } /*if*/
} /*asd_match_up_bits*/

#define B_LEFT_O_ANALYSE_NONE                     0
#define B_LEFT_O_ANALYSE_CONT                     1
#define B_LEFT_O_ANALYSE_CONT_ASSIGN_MIN          2
#define B_LEFT_O_ANALYSE_CONT_MUST_ASSIGN         3
#define B_LEFT_O_ANALYSE_DC                       4
#define B_LEFT_O_ANALYSE_DC_ASSIGN_MIN            5 /* not allowed */
#define B_LEFT_O_ANALYSE_DC_MUST_ASSIGN           6
#define B_LEFT_O_ANALYSE_DC_ELSE_CONT_ASSIGN_MIN  7
#define B_LEFT_O_ANALYSE_DC_ELSE_CONT_MUST_ASSIGN 8
#define B_LEFT_O_ANALYSE_TYPE                     4


/*
 * name
 *    asd_analyse_bits_left_over
 * purpose
 *    to analyse left over bits to see where they belong
 */
void
asd_analyse_bits_left_over(Asd_Domain *d_to_split, Asd_Domain_List *d_list,
                           Asd_Contact_Info **contact_info,
                           Asd_Contact_Info **contact_rev_info,
                           struct brookn *bn, char *ss)
{
    int i;                /* loop counter              */
    int j;                /* loop counter              */
    bool_t b_assigned;    /* bits left over assigned   */
    bool_t d_assigned;    /* domains not glob assigned */
    float corr_value;     /* correlation value         */
    float new_corr_value; /* correlation value         */
    float old_glob;       /* globularity               */
    float new_glob;       /* globularity               */
    float t_glob;         /* globularity               */
    Asd_Domain *c_domain; /* current domain            */
    Asd_Domain *c_bit;    /* current bit               */

    b_assigned = FALSE;
    d_assigned = FALSE;
    old_glob = 0.0;
    i = 1;
    j = 0;
    while (i <= d_to_split->end1) {
        j = j + contact_info[i][0].res_num;
        i++;
    } /*while*/
    c_dens = (float) j / (float) (d_to_split->end1 - d_to_split->start1 + 1);

    if (d_list->n_b_left_over == 0 ||
                               B_LEFT_O_ANALYSE_TYPE == B_LEFT_O_ANALYSE_NONE) {
        ;
    } else if (B_LEFT_O_ANALYSE_TYPE == B_LEFT_O_ANALYSE_CONT_MUST_ASSIGN ||
               B_LEFT_O_ANALYSE_TYPE == B_LEFT_O_ANALYSE_CONT ||
               B_LEFT_O_ANALYSE_TYPE == B_LEFT_O_ANALYSE_CONT_ASSIGN_MIN) {
        i = 0;
        while (i < d_list->n_b_left_over) {
            c_bit = &(d_list->bits_left_over[i]);
            j = 0;
            corr_value = -1.0;
            while (j < d_list->n_d_list) {
                c_domain = &(d_list->domains[j]);
/* if contiguous try and match up */
                if (c_bit->start1 == (c_domain->end1 + 1) ||
                    c_bit->end1 == (c_domain->start1 - 1) ||
                   (c_bit->start1 == (c_domain->end2 + 1) ||
                    c_bit->end1 == (c_domain->start2 - 1) &&
                                      c_domain->type == 2)) {
                    if (asd_match_up_bits(c_domain, c_bit, contact_info,
                                   contact_rev_info, ss, &new_corr_value, bn)) {
                        if (new_corr_value < corr_value || corr_value < 0.0) {
                                c_bit->d_assigned = j;
                                corr_value = new_corr_value;
                                old_glob = new_glob;
                        } /*if*/
                    } else if (B_LEFT_O_ANALYSE_TYPE ==
                               B_LEFT_O_ANALYSE_CONT_MUST_ASSIGN) {
                        if ((new_corr_value < corr_value || corr_value < 0.0) &&
                             new_corr_value > 0.0) {
                            c_bit->d_assigned = j;
                            corr_value = new_corr_value;
                        } /*if*/
                    } /*if*/
                } /*if*/
                j++;
            } /*while*/
            i++;
        } /*while*/
    } else if (B_LEFT_O_ANALYSE_TYPE == B_LEFT_O_ANALYSE_DC_MUST_ASSIGN ||
               B_LEFT_O_ANALYSE_TYPE == B_LEFT_O_ANALYSE_DC) {
        i = 0;
        while (i < d_list->n_b_left_over) {
            c_bit = &(d_list->bits_left_over[i]);
            j = 0;
            corr_value = -1.0;
            while (j < d_list->n_d_list) {
                c_domain = &(d_list->domains[j]);
                if (asd_match_up_bits(c_domain, c_bit, contact_info,
                                   contact_rev_info, ss, &new_corr_value, bn)) {
                    if (new_corr_value < corr_value || corr_value < 0.0) {
                            c_bit->d_assigned = j;
                            corr_value = new_corr_value;
                            old_glob = new_glob;
                    } /*if*/
                } else if (B_LEFT_O_ANALYSE_TYPE ==
                           B_LEFT_O_ANALYSE_DC_MUST_ASSIGN) {
                    if ((new_corr_value < corr_value || corr_value < 0.0) &&
                         new_corr_value > 0.0) {
                        corr_value = new_corr_value;
                        c_bit->d_assigned = j;
                    } /*if*/
                } /*if*/
                j++;
            } /*while*/
            i++;
        } /*while*/
    } /*if*/
/* now give comments to user */
    i = 0;
    while (i < d_list->n_b_left_over) {
        c_bit = &(d_list->bits_left_over[i]);
        if (c_bit->d_assigned != -1) {
            ass_bit_comments(c_bit, &(d_list->domains[c_bit->d_assigned]), bn);
        } /*if*/
        i++;
    } /*while*/
} /*asd_analyse_bits_left_over*/
