/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK is Copyright (1995) A. S. Siddiqui and G. J. Barton

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Name
 *    asd_value.c
 * Purpose
 *    to get the value of the split
 * SccsId
 *    %W%   %U%   %E%
 */

#include <ase_error.h>
#include <asd_structs.h>
#include <stdio.h>
#include <asd_value.h>
#include <asm_mop.h>
#include <asd_make_domains.h>

extern Asd_Parameters params;

Asd_Helix_Info *hel_list; /* pointer to helix list structure */
int num_hel_list;         /* number in helix list            */

void
asd_sort_helices(int d_start, int d_end, Asd_Contact_Info **contact_info,
                 char *ss)
{
    int i, j, k, x; /* integer counter */
    int num_contacts;
    int start;
    int end;
    int int1;
    int hel_ramp;
    int length, left_length;
    float area, c_dens;
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */

    hel_list = NULL;
    num_hel_list = 0;
    i = d_start;
    while (i <= d_end) {
        while (i <= d_end && ss[i - 1] != 'H') {
            i++;
        } /*while*/

        if (i == d_end) {
            i++;
        } /*if*/
        start = i;

        while (i <= d_end && ss[i - 1] == 'H') {
            i++;
        } /*while*/

        end = i - 1;

        cf1_ptr = (contact_info + start);
        j = start;

        int1 = 0;
        while (j <= end && i <= d_end) {
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            k = 0;
            while (k < num_contacts) {
                if ((*cf2_ptr).res_num >= start && (*cf2_ptr).res_num <= end) {
                    int1++;
                } /*if*/
                k++;
                cf2_ptr++;
            } /*while*/
            j++;
            cf1_ptr = (cf1_ptr + 1);
        } /*while*/

        if (int1 > 0) {
            length = end - start + 1;
            if (length >= params.MIN_HELIX_LENGTH) {
                if (length < (2 * params.HELIX_RAMP)) {
                    hel_ramp = (length / 2) + (length % 2);
                } else {
                    hel_ramp = params.HELIX_RAMP;
                } /*if*/

                left_length = length - (2 * hel_ramp);
                x = 1;
                area = 0.0;
                while (x < hel_ramp) {
                    area = 2.0 * ((float) x / (float) hel_ramp);
                    x++;
                } /*while*/
                if (left_length == -1) {
                    area = area + 1.0;
                } else {
                    area = area + 2.0 + ((float) left_length);
                } /*if*/

                c_dens = (float) int1 / area;

                if (c_dens > params.HELIX_REDUCE_C_DENS) {
                    num_hel_list++;
                    hel_list = (Asd_Helix_Info *) asm_realloc(hel_list,
                               (sizeof(Asd_Helix_Info) * num_hel_list));
                    hel_list[num_hel_list - 1].start = start;
                    hel_list[num_hel_list - 1].end = end;
                    hel_list[num_hel_list - 1].length = length;
                    hel_list[num_hel_list - 1].hel_ramp = hel_ramp;
                    hel_list[num_hel_list - 1].c_dens = c_dens -
                                                        params.HELIX_REDUCE_C_DENS;
                    hel_list[num_hel_list - 1].area = area;
                } /*if*/
            } /*if*/
        } /*if*/
    } /*while*/
} /*asd_sort_helices*/ 

void
asd_hel_reduce_ext_by(int cut_point, int *ext)
{
    int i;
    int r_start, r_end; /* end of ramp sections */
    float area;

    i = 0;
    while (i < num_hel_list) {
        if (hel_list[i].end <= cut_point || hel_list[i].start > cut_point) {
            ;
        } else {
            r_start = hel_list[i].start + hel_list[i].hel_ramp - 1;
            r_end = hel_list[i].end - hel_list[i].hel_ramp + 1;
            if (cut_point < r_start) {
                area = ((float) (hel_list[i].hel_ramp -
                                (r_start - cut_point)) /
                       ((float) hel_list[i].hel_ramp * 2.0));
            } else if (cut_point >= r_end) {
                area = ((float) (hel_list[i].hel_ramp -
                                (cut_point - r_end + 1)) /
                       ((float) hel_list[i].hel_ramp * 2.0));
            } else {
                area = 0.5;
            } /*if*/
            *ext = *ext - (int) (area * hel_list[i].c_dens);
        } /*if*/
        i++;
    } /*while*/
} /*asd_hel_reduce_ext_by*/


void
asd_hel_reduce_int_by(int d_start, int d_end, float *int_f)
{
    int i, j, k;
    int r_start, r_end; /* end of ramp sections */
    float area;

    if (d_start >= d_end) {
        ase_error_fatal("asd_hel_reduce_int_by", "routine called incorrectly");
    } /*if*/

    i = 0;
    while (i < num_hel_list) {
        if (hel_list[i].end < d_start || hel_list[i].start > d_end) {
            ;
        } else if (hel_list[i].start >= d_start && hel_list[i].end <= d_end) {
            *int_f = *int_f - (hel_list[i].area * hel_list[i].c_dens);
        } else {
            r_start = hel_list[i].start + hel_list[i].hel_ramp - 1;
            r_end = hel_list[i].end - hel_list[i].hel_ramp + 1;
            area = 0.0;
            if (hel_list[i].start <= d_start) {
                j = d_start;
                if (j < r_start && hel_list[i].start != d_start) {
                    area = - ((float) (hel_list[i].hel_ramp - (r_start - j)) /
                       ((float) hel_list[i].hel_ramp * 2.0));
                } else if (j > r_end) {
                    area = - ((float) (hel_list[i].hel_ramp - (j - r_end)) /
                       ((float) hel_list[i].hel_ramp / 2.0));
                } else {
                    area = -0.5;
                } /*if*/
            } else {
                j = hel_list[i].start;
            } /*if*/
            while (j <= r_start && j <= d_end) {
                area = area + ((float) (hel_list[i].hel_ramp - (r_start - j)) /
                       (float) hel_list[i].hel_ramp);
                j++;
            } /*while*/
            if (r_start != r_end) {
                while (j <= r_end && j <= d_end) {
                    area = area + 1.0;
                    j++;
                } /*while*/
            } /*if*/
            while (j <= hel_list[i].end && j <= d_end) {
                area = area + ((float) (hel_list[i].hel_ramp - (j - r_end)) /
                       (float) hel_list[i].hel_ramp);
                j++;
            } /*while*/
            if (hel_list[i].end > d_end) {
               j = d_end;
               if (j < r_start) {
                   area = area -
                       ((float) (hel_list[i].hel_ramp - (r_start - j)) /
                       ((float) hel_list[i].hel_ramp * 2.0));
               } else if (j > r_end) {
                   area = area - ((float) (hel_list[i].hel_ramp - (j - r_end)) /
                       ((float) hel_list[i].hel_ramp / 2.0));
                } else {
                    area = area - 0.5;
                } /*if*/
            } /*if*/

            if (hel_list[i].end == d_start || hel_list[i].start == d_end) {
                area = 0.0;
            } /*if*/
            *int_f = *int_f - (area * hel_list[i].c_dens);
        } /*if*/
        i++;
    } /*while*/
    if (*int_f <= 0.0) {
        *int_f = ZERO_SUB;
    } /*if*/
} /*asd_hel_reduce_int_by*/

float
asd_calc_c_value(int int_1, int int_2, int ext_no, int ext_sht,
                int s1_s, int s1_e, int dum1, int dum2, int s2_s, int s2_e,
                int s2_s2, int s2_e2)
{
    float int_1_f;
    float int_2_f;
    float ext_no_f;
    if (dum1 != 0 || dum2 != 0  || s1_s >= s1_e || s2_s >= s2_e) {
        ase_error_fatal("asd_get_c_value", "routine called incorrectly");
    } /*if*/

    if (int_1 == 0) {
        int_1_f = ZERO_SUB;
    } else {
        int_1_f = (float) int_1;
    } /*if*/

    if (int_2 == 0) {
        int_2_f = ZERO_SUB;
    } else {
        int_2_f = (float) int_2;
    } /*if*/

    if (ext_no == 0) {
        ext_no_f = INF_SUB;
    } else {
        ext_no_f = (float) ext_no;
    } /*if*/

    ext_no_f = (ext_no_f * ((float) ext_sht) * params.E_WEIGHT) + ext_no_f;

    asd_hel_reduce_int_by(s1_s, s1_e, &int_1_f);
    asd_hel_reduce_int_by(s2_s, s2_e, &int_2_f);
    if (s2_s2 != 0 && s2_e2 != 0) {
        if (s2_s2 >= s2_e2) {
            ase_error_fatal("asd_get_c_value", "routine called incorrectly");
        } /*if*/
        asd_hel_reduce_int_by(s2_s2, s2_e2, &int_2_f);
    } /*if*/

    return(((int_1_f / ext_no_f) * (int_2_f / ext_no_f)));
} /*asd_get_c_value*/

float
asd_calc_d_value(int int_1, int int_2, int ext_no, int ext_sht,
                Asd_Domain *d_to_split, int s1_s, int s1_e, int s2_s, int s2_e)
{
    float int_1_f;
    float int_2_f;
    float ext_no_f;

    if (int_1 == 0) {
        int_1_f = ZERO_SUB;
    } else {
        int_1_f = (float) int_1;
    } /*if*/

    if (int_2 == 0) {
        int_2_f = ZERO_SUB;
    } else {
        int_2_f = (float) int_2;
    } /*if*/

    if (ext_no == 0) {
        ext_no_f = INF_SUB;
    } else {
        ext_no_f = (float) ext_no;
    } /*if*/

    ext_no_f = (ext_no_f * ((float) ext_sht) * params.E_WEIGHT) + ext_no_f;

    asd_hel_reduce_int_by(s1_s, s1_e, &int_1_f);
/* if from a single scan */
    if (s2_s == 0 && s2_e == 0) {
        if (s1_s == d_to_split->start1 && s1_e == d_to_split->end1) {
            return(-1.0);
        } /*if*/
        if (s1_s == d_to_split->start1) {
            asd_hel_reduce_int_by(s1_e + 1, d_to_split->end1, &int_2_f);
        } else if (s1_e == d_to_split->end1) {
            asd_hel_reduce_int_by(d_to_split->start1, s1_s - 1, &int_2_f);
        } else {
            asd_hel_reduce_int_by(s1_e + 1, d_to_split->end1, &int_2_f);
            asd_hel_reduce_int_by(d_to_split->start1, s1_s - 1, &int_2_f);
        } /*if*/
    } else {
        asd_hel_reduce_int_by(s2_s, s2_e, &int_1_f);
        if (d_to_split->type == 1) {
            asd_hel_reduce_int_by(s1_e + 1, s2_s - 1, &int_2_f);
            if (s1_s != d_to_split->start1) {
                asd_hel_reduce_int_by(d_to_split->start1, s1_s - 1, &int_2_f);
            } /*if*/
            if (s2_e != d_to_split->end1) {
                 asd_hel_reduce_int_by(s2_e + 1, d_to_split->end1, &int_2_f);
            } /*if*/
        } else {
            if (s1_s == d_to_split->start1 && s1_e == d_to_split->end1 &&
                s2_s == d_to_split->start2 && s2_e == d_to_split->end2) {
                return(-1.0);
            } /*if*/
            if (s1_s != d_to_split->start1) {
                asd_hel_reduce_int_by(d_to_split->start1, s1_s - 1, &int_2_f);
            } /*if*/
            if (s1_e != d_to_split->end1) {
                asd_hel_reduce_int_by(s1_e + 1, d_to_split->end1, &int_2_f);
            } /*if*/

            if (s2_s != d_to_split->start2) {
                asd_hel_reduce_int_by(d_to_split->start2, s2_s - 1, &int_2_f);
            } /*if*/
            if (s2_e != d_to_split->end2) {
                asd_hel_reduce_int_by(s2_e + 1, d_to_split->end2, &int_2_f);
            } /*if*/
        } /*if*/
    } /*if*/

    return(((int_1_f / ext_no_f) * (int_2_f / ext_no_f)));

} /*asd_get_d_value*/

/*
 * name
 *    asd_check_falls_in_range
 * purpose
 *    to check if contact falls in range of domain
 */
bool_t
asd_check_falls_in_range(int res_num, Asd_Domain_List *d_list, int d_num)
{
    int i;

    if ((res_num >= d_list->domains[d_num].start1 &&
         res_num <= d_list->domains[d_num].end1) ||
        (d_list->domains[d_num].type == 2 &&
         res_num >= d_list->domains[d_num].start2 &&
         res_num <= d_list->domains[d_num].end2)) {
        return(TRUE);
    } /*if*/

    i = 0;
    while (i < d_list->n_d_list) {
        if (d_list->domains[i].d_assigned == d_num && d_num != i) {
            if ((res_num >= d_list->domains[i].start1 &&
                 res_num <= d_list->domains[i].end1) ||
                (d_list->domains[i].type == 2 &&
                 res_num >= d_list->domains[i].start2 &&
                 res_num <= d_list->domains[i].end2)) {
                return(TRUE);
            } /*if*/
        } /*if*/
        i++;
    } /*while*/

    i = 0;
    while (i < d_list->n_b_left_over) {
        if (d_list->bits_left_over[i].d_assigned == d_num) {
            if (res_num >= d_list->bits_left_over[i].start1 &&
                res_num <= d_list->bits_left_over[i].end1) {
                return(TRUE);
            } /*if*/
        } /*if*/
        i++;
    } /*while*/

    return(FALSE);
} /*asd_check_falls_in_range*/

/*
 * name
 *    asd_scan_domain
 * purpose
 *    scan for contacts
 */
void
asd_scan_domain(Asd_Contact_Info **contact_info,
                Asd_Contact_Info **contact_rev_info, struct brookn *bn,
                char *ss, Asd_Domain_List *d_list, int d_int, int d_ext,
                Asd_Domain *c_domain, int *int_c, int *ext_c, int *ext_sht_c,
                bool_t do_ext)
{
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int k;                      /* loop counter                          */
    int l;                      /* loop counter                          */
    int num_contacts;           /* number of contacts for that item      */
    int dummy_i;                /* dummy integer                         */
    float int_f;
    bool_t ss_only;             /* whether to use ss only                */
    bool_t cont;                /* continous                             */
    bool_t int1;
    bool_t ext1;
    Asd_Domain *domain;
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr;  /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */

    i = c_domain->start1;
    cf1_ptr = (contact_info + i);
    cr1_ptr = (contact_rev_info + i);
    while (i <= c_domain->end1) {
        cf2_ptr = *cf1_ptr;
        num_contacts = (*cf2_ptr).res_num;
        cf2_ptr++;
        j = 0;
        while (j < num_contacts) {
            if (asd_check_falls_in_range((*cf2_ptr).res_num, d_list, d_int)) {
                if ((*cf2_ptr).ctype != 'E') {
                    (*int_c)++;
                } /*if*/
            } else if (do_ext && asd_check_falls_in_range((*cf2_ptr).res_num,
                                                          d_list, d_ext)) {
                if ((*cf2_ptr).ctype != 'E') {
                    (*ext_c)++;
                } else {
                    (*ext_sht_c)++;
                } /*if*/
            } /*if*/
            j++;
            cf2_ptr++;
        } /*while*/

        if (do_ext) {
            cf2_ptr = *cr1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if (asd_check_falls_in_range((*cf2_ptr).res_num, d_list,
                                              d_ext)) {
                    if ((*cf2_ptr).ctype != 'E') {
                        (*ext_c)++;
                    } else {
                        (*ext_sht_c)++;
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
    } /*while*/

    int_f = *int_c;
    asd_hel_reduce_int_by(c_domain->start1, c_domain->end1, &int_f);
    if (c_domain->type == 2) {
        asd_hel_reduce_int_by(c_domain->start2, c_domain->end2, &int_f);
    } /*if*/
    *int_c = (int) int_f;
} /*asd_scan_domain*/
