/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was developed by Robert B. Russell

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "structs.h"

#define PDBDIRS "pdb_files"
#define DSSPDIRS "dssp_files"

/* Acts as an interactive front end to calculate contacts for a
 *   given Brookhaven (code) file or for a file of PDB format in
 *   the current directory */

main(argc, argv)
int argc;
char *argv[];
{
	int i,j,k,S;
	int start,end;

	int binary,get_pdb,get_dssp,get_code,get_parms;

	char *temp,*temp2,*code,*outcode,*fname;
	char *dsspfile,*pdbfile;
	char *file1,*file2,*file3;
	char c;
	FILE *f,*PDB,*DSSP;

	struct parameters parms;
	struct radii *VDW;
	struct parameters getpars();
	struct radii *getradii();



	/*  ASCII:
	 *   'c'=99, 'A'=60, '-'=45, ' '=32, 'T'=84 */

	k=clock();
	file1=(char*)malloc(100*sizeof(char));
	file2=(char*)malloc(100*sizeof(char));
	file3=(char*)malloc(100*sizeof(char));
	dsspfile=(char*)malloc(100*sizeof(char));
	pdbfile=(char*)malloc(100*sizeof(char));
	temp=(char*)malloc(100*sizeof(char));
	temp2=(char*)malloc(100*sizeof(char));
	fname=(char*)malloc(100*sizeof(char));
	code=(char*)malloc(100*sizeof(char));
	outcode=(char*)malloc(100*sizeof(char));


	/* the program will look for a file called `contacts.dat'
	 *  where it reads various variables in order:
	 *   CC_DISTANCE   <max distance considered between Calpha atoms>
	 *   AA_DISTANCE   <max distance considered between individual atoms>
	 *   HYDROPHOBIC   <max distance for C - C contacts>
	 *   HBOND	   <max distance considered an H-bond between N-O>
	 *   DISUL	   <max distance considered a disulphide bond>
	 *   TOOCLOSE      <too close for contacts other than H-bonds or disulphides>
	 *   MUCHTOOCLOSE  <too close for H-bonds or disulphides>
	 */
       	
	/* default values */
	parms.CC_DISTANCE=40.0; 
	parms.AA_DISTANCE=0.0; 
	parms.HYDROPHOBIC=5.0;
	parms.HBOND=3.0;
	parms.DISUL=2.2;
	parms.TOOCLOSE=2.0;
	parms.MUCHTOOCLOSE=1.2;
	parms.WATERS=0;
	parms.ACEFOR=0;
	parms.FORCE=0;
	strcpy(parms.VDW_FILE,"/data/contacts/VDW_FILE");

        binary=0;    /* set to 1 if -b option is specified */
        get_dssp=1;  /* set to 0 if a dssp file is specified */
        get_pdb=1;   /* set to 0 if a pdb file is specified */
        get_parms=0; /* set to 1 if a parameter file is given */
        get_code=1;  /* set to 0 if a code is given */
	outcode[0]='\0';

        if(argc<3) exit_error();

        for(i=1; i<argc; ++i) {
           if(argv[i][0]!='-') exit_error();
           if(argv[i][1]=='c') {
              /* PDB code given */
              strcpy(code,argv[i+1]);
              get_code=0;
              i++;
           } else if(argv[i][1]=='f') {
              /* file name */
              if((PDB=fopen(argv[i+1],"r"))==NULL) {
                printf("error: file %s does not exist\n",argv[i+1]);
                exit(-1);
              }
              fclose(PDB);
              strcpy(pdbfile,argv[i+1]);
              get_pdb=0;
              i++;
           } else if(argv[i][1]=='d') {
              if((DSSP=fopen(argv[i+1],"r"))==NULL) {
                printf("error: file %s does not exist\n",argv[i+1]);
                exit(-1);
              }
              fclose(DSSP);
              strcpy(dsspfile,argv[i+1]);
              get_dssp=0;
              i++;
           } else if(argv[i][1]=='p') {
             if((f=fopen(argv[i+1],"r"))==NULL) {
                printf("error: file %s does not exist\n",argv[i+1]);
		exit(-1);
	     }
	     printf("Parameters to be read in from %s\n",argv[i+1]);
	     i++;
	     get_parms=1;
	   } else if(argv[i][1]=='b') {
	       binary=1;
	   } else if(argv[i][1]=='w') {
	       parms.WATERS=1;
	   } else if(argv[i][1]=='a') {
	       parms.ACEFOR=1;
	   } else if(argv[i][1]=='F') {
	       parms.FORCE=1; /* force all contacts to be considered regardless of missing atoms */
	   } else if(argv[i][1]=='C') {
	     /* specify the code name for output */
	     if((i+1)>=argc) exit_error();
             strcpy(outcode,argv[i+1]);
	     i++;
	     get_code=0;
	   } else  {
	     exit_error();
	   }
	}

	if(!get_parms) {
	   printf("Using all default distances\n");
	} else {
	   parms=getpars(f,&i);
	   if(!i) { 
	      printf("error: problem with parameter file\n");
	      exit(-1);
	   }
	   fclose(f);
	}
	/* Find PDB and DSSP files if necessary */
	if(get_code) { /* read code from PDB file */
	     /* first remove any directory name */
	     start=0; end=strlen(pdbfile);
	     for(i=0; i<strlen(pdbfile); ++i) if(pdbfile[i]=='/') start=i+1;
	     if(strncmp(&pdbfile[start],"pdb",3)==0) start+=3;
	     for(i=start; i<end; ++i) 
	     if(pdbfile[i]=='.') {
		end=i-1;
		break;
	     }
	     strncpy(code,&pdbfile[start],(end-start+1));
	     code[(end-start+1)]='\0';
	  } 
	 if(get_pdb) { 
	    pdbfile=RBR_getfile(code,PDBDIRS,4,stdout);
	    if(pdbfile[0]=='\0') { 
	      printf("error: no pdbfile found for code %s\n",&argv[1][2]);
	      exit(-1);
	    }
	  } 

	if(get_dssp) { 
	   dsspfile=RBR_getfile(code,DSSPDIRS,4,stdout);
	   if(dsspfile[0]=='\0') { 
	     printf("error: no dsspfile found for code %s\n",&argv[1][2]);
	     return 0;
	   }
	}
	if(outcode[0]!='\0') strcpy(code,outcode);

	parms.SQHBOND=(parms.HBOND*parms.HBOND);
	parms.SQDISUL=(parms.DISUL*parms.DISUL);
	parms.SQTOOCLOSE=(parms.TOOCLOSE*parms.TOOCLOSE);
	parms.SQMUCHTOOCLOSE=(parms.MUCHTOOCLOSE*parms.MUCHTOOCLOSE);

	printf("VDW radii to be read in from %s\n",parms.VDW_FILE);
	if((f=fopen(parms.VDW_FILE,"r"))==NULL) {
	   printf("error: file %s not found\n",parms.VDW_FILE);
	   exit(-1);
	}
	VDW=getradii(f);
	fclose(f);

	printf("parameters:\n");
	printf(" Ca - Ca %6.3f\n",parms.CC_DISTANCE);
	printf(" Atom - Atom contact <= VDW radii + %6.3f\n",parms.AA_DISTANCE);
	printf(" C - C \"hydrophobic\" contact <= %6.3f\n",parms.HYDROPHOBIC);
	printf(" Hydrogen bond distance <= %6.3f\n",parms.HBOND);
	printf(" Disulphide bond distance <= %6.3f\n",parms.DISUL);
	printf(" Too close for contacts except H-bonds and S-S <= %6.3f\n",parms.TOOCLOSE);
	printf(" Too close for H-bonds and S-S bonds <= %6.3f\n",parms.MUCHTOOCLOSE);
	printf(" Waters are to be "); 
	if(parms.WATERS) printf("considered\n"); 
	else printf("ignored\n");
	printf(" Acetylation/Formylation atoms are to be ");
	if(parms.ACEFOR) printf("considered\n"); 
	else printf("ignored\n");

	printf("\n");


	if(binary) sprintf(file1,"%s.all.b",code);
	else sprintf(file1,"%s.all",code);
	sprintf(file2,"%s.sum",code);
	sprintf(file3,"%s.num",code);

	if((f=fopen(dsspfile,"r"))==NULL) {
	   printf("error: file %s does not exist\n",dsspfile);
	   return -1; 
	}
	fclose(f);

	printf("input:\nPDBfile: %s\n",pdbfile);
	if(S) printf("summary file: ");
	else printf("DSSP file: ");
	printf("%s\n",dsspfile);
	printf("Output:\n complete: %s; summary: %s\n",file1,file2);
	if(binary) printf("Binary mode is on\n");

	contacts(pdbfile,dsspfile,file1,file2,file3,S,VDW,binary,parms);
	  /* S states whether the SS file is a summary or a DSSP file */

	k=clock();
	printf("elapsed cpu time %10.5f seconds\n",(float)((float)k/1000000));
	exit(0);
}

int exit_error()
{
 	printf("format: contacts -c <code>\n");
        printf("   (-f <PDB file> -d <DSSP file>\n");
	printf("    -p <parameter file> -v <VDW file> -C <output code> \n");
        printf("    -b -w -a -F ) \n");
        printf("       -b ==> produce binary output\n");
	printf("       -w ==> include waters\n");
	printf("       -a ==> include ACE/FOR atoms\n");
	printf("       -F ==> force output\n");
	exit(0);
}

struct radii *getradii(vfile)
FILE *vfile;
{
	/* assumes radii are given for A -- Z, if none is present for a particular letter, 
	 *  then a value of 2.5 is assigned as a default */
	
	char atom,c;
	int i;
	float radius;
	struct radii *VDW;

	VDW=(struct radii*) malloc(26*sizeof(struct radii));

	printf("Van der-Waals radii:\n");
	for(i=0; i<26; ++i) { /* set default */
	   VDW[i].atom=(char)((int)'A'+i);
	   VDW[i].radius=2.5;
	}

	while(fscanf(vfile,"%c %f",&atom,&radius)!=(int)EOF) {
	   if(atom<'A' || atom > 'Z') {
	      printf("error: atom type '%c' not recognized\n",atom);
	      exit(-1);
	   }
	   VDW[atom-'A'].radius=radius;
	   printf("   %c %6.3f\n",atom,radius);
	   while((c=getc(vfile))!=(char)EOF && c!='\n');
	}
	return VDW;
}
