/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/* 
 * Title
 *     domak.c
 * Purpose
 *     contains the main part of the DO-M-AK program
 * SccsId:
 *     %W%   %U%   %E%
 */

#include <malloc.h>
#include <stdlib.h>
#include <array.h>
#include <stdio.h>
#include <rdssp.h>
#include <asd_structs.h>
#include <string.h>
#include <gjutil.h>
#include <asstr_util.h>
#include <ass_stamp_utils.h>
#include <asd_make_domains.h>
#include <asd_contacts_utils.h>
#include <rbr_newlib.h>
#include <asd_get_params.h>

extern Asd_Parameters params;

void
main(int argc, char *argv[])
{
    FILE *test;                          /* temporary file p        */
    FILE *output_fptr;                   /* output file pointer     */
    FILE *rasmol_fptr;                   /* rasmol file pointer     */
    int num_domains;                     /* number of domains       */
    int num_atoms;                       /* number of atoms         */
    int i;                               /* loop counter            */
    int j;                               /* loop counter            */
    int min_domain;                      /* lowest domain           */
    int max_domain;                      /* highest domain          */
    int got_code;
    int got_chain;
    int get_params;
    struct domain_loc *domain;           /* domain structure        */
    char *code;                         /* pdb code                */
    char code_id[7];                     /* pdb code                */
    char *output_file_name;              /* output file name        */
    char *rasmol_file_name;              /* rasmol file name        */
    char *dssp_file_name;                /* dssp file name          */
    char *contacts_file_name;            /* contacts file name      */
    char *pdb_file_name;                 /* pdb file name           */
    char *parameters_file_name;          /* parameters file name    */
    char *ss;                            /* secondary struct        */
    char *sheet;                         /* sheet structure         */
    char *env;                           /* enviroment variable     */
    char cid;                            /* chain identifier        */
    struct brookn *bn;                   /* brookn numbers          */
    struct br *bridges;                  /* beta sheet bridges      */
    Asd_Contact_Info **contact_info;     /* contacts structure read */
    Asd_Contact_Info **contact_rev_info; /* contacts structure read */
    Asd_Domain_List *d_list;             /* list of domains         */

    if (argc < 2) {
        fprintf(stdout, "usage : domak -c<PDB code> (-d<chain> -p<parameters file> -o<outputfile>)\n");
        exit(-1);
    } /*if*/

    if((env=getenv("DOMAKDIR"))==NULL) {
        printf("Error must set enviroment variable DOMAKDIR\n");
        exit(-1);
    } /*if*/

    GJinitfile();
    output_file_name = asstr_save("output.res");
    got_code = 0;
    got_chain = 0;
    get_params = 0; 
    cid = '*';
    i = 1;
    while(i<argc) {
        if (argv[i][0] != '-') {
            fprintf(stdout, "usage : domak -c <PDB code> (-d <chain> -p <parameters file> -o <outputfile>)\n");
            exit(-1);
        } /*if*/
        if(argv[i][1]=='c') {
            /* PDB code given */
            strcpy(code_id,&(argv[i][2]));
            got_code = 1;
        } else if (argv[i][1]=='d') {
            /*chain*/
            cid = argv[i][2];
            got_chain = 1;
        } else if (argv[i][1]=='p') {
            /*parameters file*/
            parameters_file_name = &(argv[i][2]);
            get_params = 1;
        } else if (argv[i][1]=='o') {
            /*output file*/
            output_file_name = &(argv[i][2]);
        } else {
            fprintf(stdout, "usage : domak -c <PDB code> (-d <chain> -p <parameters file> -o <outputfile>)\n");
            exit(-1);
        } /*if*/ 
        i++;
    } /*while*/
    if (got_code == 0) {
        fprintf(stdout, "usage : domak -c <PDB code> (-d <chain> -p <parameters file> -o <outputfile>)\n");
            exit(-1);
    } /*if*/

    fprintf(stdout, "# DOMAK running....\n");
    test = GJfopen(output_file_name, "a", 1);
    fclose(test);

    params.MIN_DOMAIN_SIZE          = 40;
    params.MIN_SEGMENT_SIZE_END     =  5;
    params.MIN_SEGMENT_SIZE_MID     = 25;
    params.MIN_DOUBLE_SPLIT         =120;
    params.MIN_NO_CONTACT_CUTOFF_MID= 30;
    params.MIN_NO_CONTACT_CUTOFF_END= 10;
    params.E_WEIGHT                 =  0.1;

    params.MAX_ALLOWABLE_GLOB       =  2.85;

    params.MIN_PEAK_SS_ONLY_C       = 17.05;
    params.MIN_PEAK_C               =  9.5;
    params.MIN_PEAK_SS_ONLY_DC      = 17.05;
    params.MIN_PEAK_DC              =  9.5;
    params.MIN_PEAK_SS_ONLY_MC      = 17.05;
    params.MIN_PEAK_MC              =  9.5;

    params.MIN_PEAK_BLO_C           = 60.00;
    params.MIN_PEAK_SS_ONLY_BLO_C   = 60.00;
    params.MIN_PEAK_BLO_DC          = 60.00;
    params.MIN_PEAK_SS_ONLY_BLO_DC  = 60.00;

    params.MIN_SS_PER               = 0.57;

    params.MIN_HELIX_LENGTH         =  5;
    params.HELIX_RAMP               =  4;
    params.HELIX_REDUCE_C_DENS      = 10.32;

    params.INCREMENT_DIVIDER        =250;

/* get parameters */
    if (get_params) {
        test = GJfopen(parameters_file_name, "r", 1);
        asd_get_params(test);
        fclose(test);
    } else {
        fprintf(stdout, "# Using default parameters\n");
    } /*if*/
/* write out parameters */

    fprintf(stdout, "# MIN_DOMAIN_SIZE %d\n", params.MIN_DOMAIN_SIZE);
    fprintf(stdout, "# MIN_SEGMENT_SIZE_END %d\n", params.MIN_SEGMENT_SIZE_END);
    fprintf(stdout, "# MIN_SEGMENT_SIZE_MID %d\n", params.MIN_SEGMENT_SIZE_MID);
    fprintf(stdout, "# MIN_DOUBLE_SPLIT %d\n", params.MIN_DOUBLE_SPLIT);
    fprintf(stdout, "# MIN_NO_CONTACT_CUTOFF_MID %d\n", params.MIN_NO_CONTACT_CUTOFF_MID);
    fprintf(stdout, "# MIN_NO_CONTACT_CUTOFF_END %d\n", params.MIN_NO_CONTACT_CUTOFF_END);
    fprintf(stdout, "# E_WEIGHT %f\n", params.E_WEIGHT);
    fprintf(stdout, "# MAX_ALLOWABLE_GLOB %f\n", params.MAX_ALLOWABLE_GLOB);
    fprintf(stdout, "# MIN_PEAK_SS_ONLY_C %f\n", params.MIN_PEAK_SS_ONLY_C);
    fprintf(stdout, "# MIN_PEAK_C %f\n", params.MIN_PEAK_C);
    fprintf(stdout, "# MIN_PEAK_SS_ONLY_DC %f\n", params.MIN_PEAK_SS_ONLY_DC);
    fprintf(stdout, "# MIN_PEAK_DC %f\n", params.MIN_PEAK_DC);
    fprintf(stdout, "# MIN_PEAK_SS_ONLY_MC %f\n", params.MIN_PEAK_SS_ONLY_MC);
    fprintf(stdout, "# MIN_PEAK_MC %f\n", params.MIN_PEAK_MC);
    fprintf(stdout, "# MIN_PEAK_BLO_C %f\n", params.MIN_PEAK_BLO_C);
    fprintf(stdout, "# MIN_PEAK_SS_ONLY_BLO_C %f\n", params.MIN_PEAK_SS_ONLY_BLO_C);
    fprintf(stdout, "# MIN_PEAK_BLO_DC %f\n", params.MIN_PEAK_BLO_DC);
    fprintf(stdout, "# MIN_PEAK_SS_ONLY_BLO_DC %f\n", params.MIN_PEAK_SS_ONLY_BLO_DC);
    fprintf(stdout, "# MIN_SS_PER %f\n", params.MIN_SS_PER);
    fprintf(stdout, "# MIN_HELIX_LENGTH %d\n", params.MIN_HELIX_LENGTH);
    fprintf(stdout, "# HELIX_RAMP %d\n", params.HELIX_RAMP);
    fprintf(stdout, "# HELIX_REDUCE_C_DENS %f\n", params.HELIX_REDUCE_C_DENS);
    fprintf(stdout, "# INCREMENT_DIVIDER %d\n", params.INCREMENT_DIVIDER);

/* call function that reads and checks domains */

    code = asstr_cat_safe(asstr_save(code_id), asstr_save("-"));

    dssp_file_name = RBR_getfile(code_id,
                 asstr_cat_safe(asstr_save(env), "/dssp_files"), 4, stdout);
    contacts_file_name = RBR_getfile(code_id,
                 asstr_cat_safe(asstr_save(env), "/contacts_files"), 4, stdout);
    pdb_file_name = RBR_getfile(code_id,
                 asstr_cat_safe(asstr_save(env), "/pdb_files"), 4, stdout);
    fprintf(stdout, "# pdb file : %s\n", pdb_file_name);
    fprintf(stdout, "# dssp file : %s\n", dssp_file_name);
    fprintf(stdout, "# contacts file : %s\n", contacts_file_name);
    test = GJfopen(dssp_file_name, "r", 1);
    fclose(test);
    test = GJfopen(contacts_file_name, "r", 1);
    fclose(test);
    test = GJfopen(pdb_file_name, "r", 1);
    fclose(test);
    rasmol_file_name = asstr_cat_safe(asstr_save(code_id),
                                      asstr_save(".rasmol"));
    test = GJfopen(rasmol_file_name, "w", 1);
    fclose(test);

    fprintf(stdout, "# Reading in contacts file....\n");
    fflush(stdout);
    asd_read_all_info(contacts_file_name, dssp_file_name, &num_atoms,
                      &bn, &ss, &bridges, &sheet, &contact_info,
                      &contact_rev_info);

/*
 * ************ cid temporary argument in design stage
 */
    ass_set_code(code);
    ass_set_pdb_file(pdb_file_name);
    if (cid == '*') {
        fprintf(stdout, "# Analysing all residues in %s\n", code_id);
    } else {
        fprintf(stdout, "# Analysing residues in chain %c of %s\n",
                cid, code_id);
    } /*if*/
    d_list = asd_work_out_domains(num_atoms, contact_info,
                                  contact_rev_info, bn, cid, ss);

    asd_print_domains(d_list, bn, stdout, NULL, cid);
    output_fptr = GJfopen(output_file_name, "a", 1);
    rasmol_fptr = GJfopen(rasmol_file_name, "w", 1);
    asd_print_domains(d_list, bn, output_fptr, rasmol_fptr, cid);
    fclose(output_fptr);
    fclose(rasmol_fptr);
    free(bn);
    free(ss);
    free(bridges);
    free(sheet);
    j = 0;
    while (j <= num_atoms) {
        free(contact_info[j]);
        free(contact_rev_info[j]);
        j++;
    } /*while*/
    free(contact_info);
    free(contact_rev_info);

} /*main*/
